import { load } from 'cheerio';
import QuickLRU from 'quick-lru';

const blockMap = {
  "wp-audio-shortcode": "audio",
  "wp-block-audio": "audio",
  "wp-block-file": "file",
  "wp-block-attachment": "file",
  "wp-block-video": "video",
  "guaqi-form": "form"
};
const decode = (() => {
  const cache = new QuickLRU({ maxSize: 1e3 });
  return (str) => {
    if (!str) return "";
    if (cache.has(str)) return cache.get(str);
    try {
      const v = decodeURIComponent(str);
      cache.set(str, v);
      return v;
    } catch (error) {
      return str;
    }
  };
})();
const htmlToBlocks = (rawHtml) => {
  if (!rawHtml || typeof rawHtml !== "string") return [];
  const $ = load(rawHtml, {
    recognizeSelfClosing: true,
    decodeEntities: true,
    sourceCodeLocationInfo: true
  });
  const blocks = [];
  let paragraphAcc = [];
  const flushParagraph = () => {
    const html = paragraphAcc.join("").trim();
    if (html) blocks.push({ type: "paragraph", html });
    paragraphAcc.length = 0;
  };
  $("body").children().each((_, el) => {
    var _a, _b;
    const $el = $(el);
    const cls = Object.keys(blockMap).find(
      (c) => $el.hasClass(c) || $el.find("div,a").hasClass(c)
    );
    if (cls) {
      flushParagraph();
      const src = $el.attr("src") || $el.find("a,div").attr("href") || $el.find("audio,video,a,source,attachment").attr("src") || "";
      const type = blockMap[cls];
      const block = { type, src };
      switch (type) {
        case "audio":
          block.data = [
            {
              src,
              thumb: $el.attr("poster") || $el.find("img").attr("src") || "",
              name: decode(
                $el.attr("title") || $el.find("figcaption").text().trim() || ((_a = src.split("/").pop()) == null ? void 0 : _a.split(".")[0]) || ""
              )
            }
          ];
          break;
        case "video":
          block.data = [
            {
              poster: $el.attr("poster") || $el.find("img").attr("src") || "",
              muted: $el.attr("muted") !== void 0,
              src
            }
          ];
          break;
        case "form":
          block.data = [
            {
              id: $el.attr("data-id") || "",
              size: $el.attr("data-size") || "small"
            }
          ];
          break;
        case "file":
          block.data = [
            {
              src,
              name: decode(
                $el.attr("title") || $el.find("figcaption").text().trim() || $el.find("attachment").attr("title") || ((_b = src.split("/").pop()) == null ? void 0 : _b.split(".")[0]) || ""
              ),
              unzipPassword: $el.find("attachment").attr("unzippassword") || $el.attr("data-unzip-password"),
              extractionCode: $el.find("attachment").attr("extractioncode") || $el.attr("data-extraction-code"),
              size: $el.find("attachment").attr("filesize") || $el.attr("data-size"),
              ext: $el.find("attachment").attr("ext") || src.split(".").pop()
            }
          ];
          break;
      }
      blocks.push(block);
    } else {
      const loc = el.sourceCodeLocation;
      paragraphAcc.push(
        loc ? rawHtml.slice(loc.startOffset, loc.endOffset) : $.html(el)
      );
    }
  });
  flushParagraph();
  $.root().empty();
  return blocks;
};

export { htmlToBlocks as h };
