import { x as defineWebSocketHandler, u as useRuntimeConfig } from '../../_/nitro.mjs';
import crypto from 'crypto';
import QuickLRU from 'quick-lru';
import 'node:crypto';
import 'node:http';
import 'node:https';
import 'node:events';
import 'node:buffer';
import 'ioredis';
import 'node:fs';
import 'node:path';
import '@intlify/utils';
import 'vue-router';
import 'node:url';
import 'undici';
import 'turndown';
import '@joplin/turndown-plugin-gfm';
import 'unhead/server';
import 'unhead/plugins';
import 'unhead/utils';
import 'vue';
import 'vue-bundle-renderer/runtime';
import 'vue/server-renderer';

const config = useRuntimeConfig();
const MAX_SOCKETS = 1e5;
const userSocketMap = new QuickLRU({ maxSize: MAX_SOCKETS });
const HEARTBEAT_INTERVAL = 3e4;
const HEARTBEAT_TIMEOUT = 12e4;
setInterval(() => {
  try {
    const now = Date.now();
    const timeoutUsers = [];
    for (const [userId, peer] of userSocketMap.entries()) {
      if (peer.lastPong && now - peer.lastPong > HEARTBEAT_TIMEOUT) {
        timeoutUsers.push({ userId, peer });
      }
    }
    for (const { userId, peer } of timeoutUsers) {
      try {
        peer.close(1e3, "timeout");
      } catch (e) {
        console.error("[WS] Error closing peer:", e);
      }
      userSocketMap.delete(userId);
    }
  } catch (error) {
    console.error("[WS] Heartbeat error:", error);
  }
}, HEARTBEAT_INTERVAL);
const ws = defineWebSocketHandler({
  async message(peer, message) {
    try {
      const msg = JSON.parse(message);
      await handleMessage(peer, msg);
    } catch (e) {
      console.error("[WS] parse error:", e);
      try {
        peer.send(JSON.stringify({ error: "Invalid message format" }));
      } catch (err) {
        console.error("[WS] Error sending error message:", err);
      }
    }
  },
  async close(peer) {
    const userId = peer.userId;
    if (!userId) return;
    try {
      const leaveMsg = JSON.stringify({
        type: "leave_room",
        room: peer.userRoom,
        data: {
          action: "leave_room",
          user: {
            name: peer.userName,
            databaseId: userId,
            avatar: { url: peer.userAvatar }
          }
        }
      });
      try {
        if (peer.userRoom) {
          peer.publish(peer.userRoom, leaveMsg);
        }
      } catch (e) {
        console.error("[WS] Error publishing leave message:", e);
      }
    } catch (error) {
      console.error("[WS] Error in close handler:", error);
    } finally {
      userSocketMap.delete(userId);
    }
  }
});
async function handleMessage(peer, msg) {
  var _a;
  const { type, room, to, data } = msg;
  const safeSend = (message) => {
    try {
      peer.send(JSON.stringify(message));
    } catch (e) {
      console.error("[WS] Error sending message:", e);
    }
  };
  const safePublish = (room2, message) => {
    try {
      peer.publish(room2, JSON.stringify(message));
    } catch (e) {
      console.error("[WS] Error publishing message:", e);
    }
  };
  switch (type) {
    case "connect":
      if (data.databaseId && expectedSignature(data.databaseId + "_" + data.timestamp, data.signature)) {
        if (userSocketMap.has(data.databaseId)) {
          const oldPeer = userSocketMap.get(data.databaseId);
          try {
            oldPeer.close(1e3, "new_connection");
          } catch (e) {
            console.error("[WS] Error closing old peer:", e);
          }
        }
        peer.userId = data.databaseId;
        peer.userName = data.name;
        peer.userAvatar = ((_a = data.avatar) == null ? void 0 : _a.url) || data.avatar || "";
        peer.lastPong = Date.now();
        userSocketMap.set(data.databaseId, peer);
        safeSend({ type: "connect_success", userId: data.databaseId });
      } else {
        safeSend({ error: "Invalid signature or missing data" });
      }
      break;
    case "ping":
      peer.lastPong = Date.now();
      safeSend({ type: "pong" });
      break;
    case "show_room":
      try {
        const roomName = room || "chat";
        peer.subscribe(roomName);
        safeSend({ type: "show_room", room: roomName, data: [] });
      } catch (e) {
        console.error("[WS] Error in show_room:", e);
      }
      break;
    case "delete_chat":
      try {
        const [deleteSig, ts] = data.signature.split("_");
        if (expectedSignature(data.id + "_" + ts, deleteSig)) {
          safePublish("chat", { type: "delete_chat", room: "chat", data: { id: data.id } });
        }
      } catch (e) {
        console.error("[WS] Error in delete_chat:", e);
      }
      break;
    case "chat_room":
      if (room !== "chat") {
        safeSend({ error: "Room not valid" });
        return;
      }
      try {
        if (expectedSignature(data.user.databaseId + data.message + data.created_at, data.signature)) {
          const m = { type: "chat_room", room, data };
          safePublish(room, m);
          safeSend(m);
        }
      } catch (e) {
        console.error("[WS] Error in chat_room:", e);
      }
      break;
    case "private_message":
      if (!to) {
        safeSend({ error: "Recipient required" });
        return;
      }
      try {
        if (expectedSignature(peer.userId + "" + to.databaseId + data.timestamp + data.content, data.signature)) {
          sendPrivateMessage(peer, to, data);
        }
      } catch (e) {
        console.error("[WS] Error in private_message:", e);
      }
      break;
    case "join_room":
      if (room !== "chat") {
        safeSend({ error: "Room invalid" });
        return;
      }
      try {
        if (expectedSignature(peer.userId + "_" + data.timestamp, data.signature)) {
          peer.userRoom = room;
          delete data.signature;
          safeSend({ type: "join_success", isJoin: room, room, data });
          safePublish(room, { type: "join_success", room, data });
        }
      } catch (e) {
        console.error("[WS] Error in join_room:", e);
      }
      break;
    default:
      safeSend({ error: "Unknown message type" });
  }
}
function expectedSignature(key, signature) {
  const expect = crypto.createHmac("sha256", config.private.apiSecret).update(key).digest("hex");
  return expect === signature;
}
async function sendPrivateMessage(from, to, data) {
  try {
    const target = userSocketMap.get(to.databaseId);
    const msg = {
      type: "private_message",
      to,
      from: {
        databaseId: from.userId,
        name: from.userName,
        avatar: { url: from.userAvatar }
      },
      msg: data.content,
      timestamp: data.timestamp
    };
    if (target) {
      try {
        target.send(JSON.stringify(msg));
      } catch (e) {
        console.error("[WS] Error sending to target:", e);
      }
    }
    try {
      from.send(JSON.stringify(msg));
    } catch (e) {
      console.error("[WS] Error sending to sender:", e);
    }
  } catch (error) {
    console.error("[WS] Error in sendPrivateMessage:", error);
  }
}

export { ws as default };
